<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Countries;
use App\Models\TransferRoutes;
use App\Models\Cities;
use App\Models\TransferBooking;
use App\Models\RoutesWithPrice;
use App\Models\RoutesPriceDetails;
use App\Models\VechileDetails;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class CitiesController extends Controller
{
    
    /**

     * @OA\Get(
     *   path="/api/cities",
     *   tags={"Dashboard"},
     *   summary="To get cities",
     *   description="It will return an cities based on country",
     *   @OA\Parameter(
     *      name="country_id",
     *      in="query",
     *      required=true,
     *      description="Country id",
     *      @OA\Schema( type="integer")
     *    ),
     *   @OA\Parameter(
     *      name="is_favourite",
     *      in="query",
     *      description="favourite",
     *      @OA\Schema( type="integer")
     *    ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public static function cities (Request $request) 
    {
        $cities = [];
        $county = Countries::find($request->get('country_id'));
        if (@$county) {
            $cities = $county->cities;
            if (@$request->get('is_favourite')) {
                $cities = $county->cities->where('is_favourite', 1);
            }
            $cities = $cities->pluck('name', 'id')->toArray();
        }
        return response()->json(['cities' => $cities]);
    }


    /**
     * @OA\Get(
     *   path="/api/countries/{country}/{city}",
     *   tags={"Dashboard"},
     *   summary="To get places",
     *   description="It will return an places based on country and city",
     *   @OA\Parameter(
     *      name="country",
     *      in="path",
     *      required=true,
     *      description="country id",
     *      @OA\Schema( type="integer")
     *    ),
     *    @OA\Parameter(
     *      name="city",
     *      in="path",
     *      required=true,
     *      description="city id",
     *      @OA\Schema( type="integer")
     *    ),
     *    @OA\Parameter(
     *      name="categories",
     *      in="query",
     *      description="1 - popular places, 2 - recommended places",
     *      @OA\Schema( type="integer")
     *    ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public static function countries (Request $request, $county_id, $city_id) 
    {
        $CitiesControllerClass = new CitiesController();
        $cities_array = Cities::where('country_id', $county_id)->pluck('name', 'id')->toArray();
        $cities = TransferRoutes::where('from', $city_id);
        $categories = @$request->get('categories');
        if ($categories == 1) {
            $cities = $cities->where('is_popular', 1);
        } else if ($categories == 2) {
            $cities = $cities->where('is_recommended', 1);
        }
        $cities = $cities->get()->map(function($routes) use ($cities_array, $CitiesControllerClass) {
            $data['id'] = $routes->id;
            $data['from'] = $cities_array[$routes->from];
            $data['to'] = $cities_array[$routes->to];
            //$data['city_image'] = @$routes->city_image ? $CitiesControllerClass->imgToBase64($routes->city_image) : '';
            $data['city_image'] = @$routes->city_image ? 'https://admin.larkholidays.com/public/'.$routes->city_image : '';
            $route_with_prices = RoutesWithPrice::select('id')->where('route_id', $routes->id)->pluck('id')->toArray();
            $plus_two_date = date('Y-m-d', strtotime(date('Y-m-d') . ' +2 days')); // Add 2 days
            $data['route_with_min_price'] = RoutesPriceDetails::select('price')->whereIn('routes_with_price_id', $route_with_prices)->where('from_date', '<=', $plus_two_date)->where('to_date', '>=', $plus_two_date)->get()->min('price');
            return $data;
        });
        return response()->json($cities);
    }


     /**
     * @OA\Get(
     *   path="/api/vechile-search",
     *   tags={"Search page"},
     *   summary="To get vechile details",
     *   description="It will return an vechile based on from and to date",
     *    @OA\Parameter(
     *      name="from",
     *      required=true,
     *      in="query",
     *      description="From city",
     *      @OA\Schema( type="string")
     *    ),
     *    @OA\Parameter(
     *      name="to",
     *      required=true,
     *      in="query",
     *      description="To city",
     *      @OA\Schema( type="string")
     *    ), 
     *    @OA\Parameter(
     *      name="date",
     *      in="query",
     *      description="YYYY-MM-DD",
     *      @OA\Schema( type="string")
     *    ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public static function vechileSearch (Request $request) 
    {
        $CitiesControllerClass = new CitiesController();
        $cities_value = Cities::whereIn('name', [$request->from, $request->to])->pluck('id', 'name')->toArray();
        //dd($cities_value);
        $route_id = TransferRoutes::where(['from' => $cities_value[$request->from], 'to' =>  $cities_value[$request->to]])->first()->id;
        $route_with_prices = RoutesWithPrice::where('route_id', $route_id)->get()->toArray();
        $data = [];
        if (@$route_with_prices) {
            foreach ($route_with_prices as $route_with_price) {
                $result_data          = $route_with_price;
                $vechile_detail       = VechileDetails::whereId($route_with_price['vechile_id'])->where('passagner_count', '>=', $request->passagner_count)->first()->toArray();
                $result_data['from_city'] = $cities_value[$request->from];
                $result_data['to_city'] = $cities_value[$request->to];
                $result_data['vechile_name'] = @$vechile_detail['name'];
                $result_data['from_city_text'] = $request->from;
                $result_data['to_city_text'] = $request->to;
                //$result_data['vechile_image'] = @$vechile_detail['vechile_image'] ? $CitiesControllerClass->imgToBase64($vechile_detail['vechile_image']) : '';
                $result_data['vechile_image'] = @$vechile_detail['vechile_image'] ? 'https://admin.larkholidays.com/public/'.$vechile_detail['vechile_image'] : '';
                $RoutesPriceDetails = RoutesPriceDetails::select('price')->where('routes_with_price_id', @$route_with_price['id']);
                if (@$request->date) {
                    $RoutesPriceDetails = $RoutesPriceDetails->where('from_date', '<=', $request->date)->where('to_date', '>=', $request->date);
                }
                if (@$RoutesPriceDetails->first()->price) {
                    $result_data['price'] = $RoutesPriceDetails->first()->price;
                    $data[] = $result_data;
                }
            }
        }
        return response()->json(['status' => 'success', 'data' => $data]);
    }


    /**
     * @OA\Post(
     *   path="/api/booking",
     *   tags={"Booking"},
     *   summary="To store or update the booking data",
     *   description="It will return a booking details id with encrpty",
     *   @OA\RequestBody(
     *         @OA\MediaType(
     *          mediaType="application/json",
     *          @OA\Schema(
     *             @OA\Property(property="id", type="integer", example=""),
     *             @OA\Property(property="routes_with_price_id", type="integer", example=""),
     *             @OA\Property(property="from", type="integer", example=""),
     *             @OA\Property(property="to", type="integer", example=""),
     *             @OA\Property(property="from_pickup", type="string", example=""),
     *             @OA\Property(property="to_drop", type="string", example=""),
     *             @OA\Property(property="from_lat", type="string", example=""),
     *             @OA\Property(property="to_lat", type="string", example=""),
     *             @OA\Property(property="date", type="string", example="YYYY-MM-DD"),
     *             @OA\Property(property="adult", type="integer", example=""),
     *             @OA\Property(property="child", type="integer", example=""),
     *             @OA\Property(property="infint", type="integer", example=""),
     *             @OA\Property(property="is_comfirm", type="integer", example=""),
     *             @OA\Property(property="supplier_id", type="integer", example=""),
     *          ),
     *   ),
     * ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public static function booking (Request $request) 
    {
        if (@$request->id) {
            $response = TransferBooking::whereId(Crypt::decrypt($request->id))->update(Arr::except($request->all(), ['id']));
        } else {
            $response = TransferBooking::create($request->all());
        }
        return response()->json([
            'status' => 'success',
            'id' => @$request->id ? $request->id : Crypt::encrypt($response->id)
        ]);
    }

    /**
     * @OA\Get(
     *   path="/api/get-booking-detail",
     *   tags={"Booking"},
     *   summary="To get booking details",
     *   description="It will return an booking detail based on user given encrpty id",
     *    @OA\Parameter(
     *      name="id",
     *      required=true,
     *      in="query",
     *      description="Enter your encrpty id",
     *      @OA\Schema( type="string")
     *    ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public static function getBookingDetails (Request $request) 
    {
        $response = [];
        if (@$request->id) {
            $response = TransferBooking::whereId(Crypt::decrypt($request->id))->first()->toArray();
            $response['from_city'] = Cities::select('name')->whereId($response['from'])->first()->name;
            $response['to_city'] = Cities::select('name')->whereId($response['to'])->first()->name;
        }
        return response()->json([
            'status' => 'success',
            'data' => @$response
        ]);
    }

    public static function imgToBase64 ($file_path) {
        $image = file_get_contents(storage_path($file_path));
        $base64Image = base64_encode($image);
        $type  = explode('.', $file_path);
        return 'data:image/'.$type[1].';base64,'.$base64Image;
    }
     /**
     * @OA\Get(
     *   path="/api/to-cities",
     *   tags={"Search page"},
     *   summary="to get the cities",
     *   description="It will return a to cities list",
     *    @OA\Parameter(
     *      name="form_city",
     *      required=true,
     *      in="query",
     *      description="Enter your encrpty city",
     *      @OA\Schema( type="string")
     *    ),
     *   @OA\Response(response=200, description="Successful operation"),
     *   @OA\Response(response=400, description="Bad request"),
     *   @OA\Response(response=401, description="Unauthorized"),
     *   @OA\Response(response=404, description="Not found"),
     *   @OA\Response(response=406, description="Not acceptable"),
     *   @OA\Response(response=422, description="Validation error"),
     *   @OA\Response(response=500, description="Internal server error")
     * )
     * 
     */
    public static function toCities (Request $request) {
        
        $toCities = DB::table('transfer_transfer_routes')->where('from', $request->from)->get();
        $data = response()->json([
            'status' => 'success',
            'data' => @$toCities
        ]);
        return $data;

    }
}